import time
import math
from ipycanvas import MultiCanvas
from IPython.display import display

class IPythonturtle:
    def __init__(self, width=500, height=500):
        # MultiCanvas mit 2 Layern: Hintergrund (Linien) + Vordergrund (Turtle)
        self.canvas = MultiCanvas(2, width=width, height=height)
        self.bg = self.canvas[0]   # Linien
        self.fg = self.canvas[1]   # Turtle-Symbol
        display(self.canvas)

        # State
        self.width_px = width
        self.height_px = height
        self.x = width / 2
        self.y = height / 2
        self.heading = 0  # 0° = rechts
        self.pen_down = True
        self.pen_color = "black"
        self.line_width = 2

        # Geschwindigkeit
        self._delay = 0.02  # Sekunden

        # Füllung
        self.fill_active = False
        self.fill_color = "yellow"
        self.fill_path = []

        self._draw_turtle()

    # --- Geschwindigkeit ---
    def speed(self, value="normal"):
        """Setzt die Zeichen-Geschwindigkeit."""
        speeds = {
            #"fastest": 0,005
            #"fast": 0.01,
            "normal": 0.02,
            "slow": 0.05,
            "slowest": 0.2
        }
        if isinstance(value, (int, float)):
            self._delay = max(0, value/1000)  # ms -> s
        else:
            self._delay = speeds.get(value, 0.02)

    # --- Bewegung ---
    def forward(self, distance):
        steps = max(1, int(abs(distance) / 2))  # Schrittgröße ~2px
        step_len = distance / steps
        rad = math.radians(self.heading)

        for _ in range(steps):
            new_x = self.x + math.cos(rad) * step_len
            new_y = self.y + math.sin(rad) * step_len
            if self.pen_down:
                self.bg.stroke_style = self.pen_color
                self.bg.line_width = self.line_width
                self.bg.begin_path()
                self.bg.move_to(self.x, self.y)
                self.bg.line_to(new_x, new_y)
                self.bg.stroke()
            self.x, self.y = new_x, new_y
            if self.fill_active:
                self.fill_path.append((self.x, self.y))
            self._draw_turtle()
            if self._delay > 0:
                time.sleep(self._delay)

    def backward(self, distance):
        self.forward(-distance)

    def left(self, angle):
        self.heading -= angle
        self._draw_turtle()

    def right(self, angle):
        self.heading += angle
        self._draw_turtle()

    def goto(self, x, y):
        dx, dy = x - self.x, y - self.y
        dist = math.hypot(dx, dy)
        steps = max(1, int(dist / 2))
        step_x = dx / steps
        step_y = dy / steps
        for _ in range(steps):
            new_x = self.x + step_x
            new_y = self.y + step_y
            if self.pen_down:
                self.bg.stroke_style = self.pen_color
                self.bg.line_width = self.line_width
                self.bg.begin_path()
                self.bg.move_to(self.x, self.y)
                self.bg.line_to(new_x, new_y)
                self.bg.stroke()
            self.x, self.y = new_x, new_y
            if self.fill_active:
                self.fill_path.append((self.x, self.y))
            self._draw_turtle()
            if self._delay > 0:
                time.sleep(self._delay)

    def home(self):
        self.goto(self.width_px/2, self.height_px/2)
        self.heading = 0
        self._draw_turtle()

    # --- Formen ---
    def circle(self, radius):
        steps = max(20, int(abs(radius) * 0.5))
        step_len = 2 * math.pi * radius / steps
        step_angle = 360 / steps
        for _ in range(steps):
            self.forward(step_len)
            self.left(step_angle)

    # --- Pen ---
    def penup(self): self.pen_down = False
    def pendown(self): self.pen_down = True
    def pencolor(self, c): self.pen_color = c
    def width(self, w): self.line_width = w

    # --- Füllen ---
    def fillcolor(self, c):
        self.fill_color = c

    def begin_fill(self):
        self.fill_active = True
        self.fill_path = [(self.x, self.y)]

    def end_fill(self):
        if self.fill_active and len(self.fill_path) > 2:
            self.bg.fill_style = self.fill_color
            self.bg.begin_path()
            self.bg.move_to(*self.fill_path[0])
            for px, py in self.fill_path[1:]:
                self.bg.line_to(px, py)
            self.bg.close_path()
            self.bg.fill()
        self.fill_active = False
        self.fill_path = []

    # --- Zeichenfläche ---
    def clear(self):
        self.bg.clear()
        self.fg.clear()
        self.home()

    # --- Turtle-Symbol ---
    def _draw_turtle(self):
        self.fg.clear()
        size = 10
        rad = math.radians(self.heading)
        pts = []
        for angle in [0, 140, -140]:
            a = rad + math.radians(angle)
            pts.append((self.x + math.cos(a) * size,
                        self.y + math.sin(a) * size))
        self.fg.fill_style = "green"
        self.fg.begin_path()
        self.fg.move_to(*pts[0])
        self.fg.line_to(*pts[1])
        self.fg.line_to(*pts[2])
        self.fg.close_path()
        self.fg.fill()
